import asyncio
import os

from datetime import datetime

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar

from predefined_tasks.common.helper import send_to_gc

from config_enum import hal_enum as hal_config

from virtualunits.HAL import HAL
from virtualunits.vu_node_application import VUNodeApplication
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.VirtualTemperatureUnit import VirtualTemperatureUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from fleming.rbartz.flash_lum_test import flash_lum_get_led_power

hal_unit: HAL = VUnits.instance.hal
eef_unit: VUNodeApplication = hal_unit.nodes['EEFNode']
fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']
meas_unit: VUMeasurementUnit = hal_unit.measurementUnit
pmt1_cooling: VirtualTemperatureUnit = hal_unit.pmt_ch1_Cooling

report_path = hal_unit.configManager.get_config(hal_config.Application.GCReportPath)
report_file = os.path.join(report_path, 'xiu20_test.csv')

# XIU20 - New Analog Measurement Electronics Tests #####################################################################

ANALOGCONTROLMODE_ADD_TO_HIGH_OFFSET            = 1
ANALOGCONTROLMODE_ADD_TO_LOW_OFFSET             = 2
ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET    = 3

async def load_xiu20_operation(op_id, meas_time_ms, repeats=1, interval_ms=0.0):
    sequence = xiu20_measurement(meas_time_ms, repeats, interval_ms)
    meas_unit.resultAddresses[op_id] = range(0, (repeats * 5))
    await meas_unit.LoadTriggerSequence(op_id, sequence)


def xiu20_measurement(meas_time_ms, repeats, interval_ms):
    if (meas_time_ms > 10000):
        raise ValueError(f"meas_time_ms must be smaller or equal to 10000 ms")
    if (repeats < 1) or (repeats > 819):
        raise ValueError(f"repeats must be in the range [1, 819]")
    if (repeats > 1) and (interval_ms < meas_time_ms):
        raise ValueError(f"interval_ms must be greater or equal to meas_time_ms")
    if (interval_ms > 43980465):
        raise ValueError(f"interval_ms must be smaller or equal to 43980465 ms")

    ignore_range = False
    offset_correction = False
    analog_offset = 1000
    
    full_reset_delay = 40000    # 400 us
    reset_low_delay = 1000      #  10 us
    reset_off_delay = 1000      #  10 us
    sample_offset_delay = 2     #  20 ns
    pre_cnt_window = 100        #   1 us
    conversion_delay = 1200     #  12 us

    meas_time_us = round(meas_time_ms * 1000)
    meas_time_us_coarse, meas_time_us_fine = divmod(meas_time_us, 65536)

    dead_time = full_reset_delay + reset_low_delay + reset_off_delay + sample_offset_delay + pre_cnt_window + conversion_delay  # 433.02 us

    interval_delay = round(interval_ms * 1e5) - round(meas_time_us * 100) - dead_time
    interval_delay_coarse, interval_delay_fine = divmod(interval_delay, 2**26)

    seq_gen = meas_seq_generator()

    # results = [pmt1_cnt, pmt1_al, pmt1_ah, pmt1_alo, pmt1_aho, ...]
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(repeats * 5)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)

    seq_gen.LoadDataReg(stackNotRegDst=False, dstReg=0, value=(-analog_offset & 0xFFFF))

    seq_gen.SetSignals(OutputSignal.InputGatePMT1)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)

    seq_gen.Loop(repeats)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    # Reset Low + High Integrators (Reset=High, Range=High) -> Integrator Mode := Full Reset
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(reset_low_delay)
    # Reset Low Integrator (Reset=Low, Range=High) -> Integrator Mode := Integrate in High Range
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)

    seq_gen.TimerWaitAndRestart(reset_off_delay)
    # Reset Off (Reset=Low, Range=Low) -> Integrator Mode := Integrate in Low Range
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)

    seq_gen.TimerWaitAndRestart(sample_offset_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if meas_time_us_coarse > 0:
        seq_gen.Loop(meas_time_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if meas_time_us_fine > 0:
        seq_gen.Loop(meas_time_us_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=4)
    if offset_correction:
        seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset)
        seq_gen.SetAnalogControl(pmt1=ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET)
    
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=ignore_range, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=ignore_range, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=2)

    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=5)

    if interval_delay_coarse > 0:
        seq_gen.Loop(interval_delay_coarse)
        seq_gen.TimerWaitAndRestart(2**26)
        seq_gen.LoopEnd()
    if interval_delay_fine > 0:
        seq_gen.TimerWaitAndRestart(interval_delay_fine)
    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    return seq_gen.currSequence


async def xiu20_test(meas_time_ms=1000):

    from predefined_tasks.pmt_adjust import set_led_current

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    op_id = 'flash_lum'
    meas_unit.ClearOperations()
    await load_xiu20_operation(op_id, meas_time_ms)
    
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    await send_to_gc(f" ")
    await send_to_gc(f"pmt1_cnt: {results[0]} ; pmt1_al: {results[1]} ; pmt1_ah: {results[2]} ; pmt1_alo: {results[3]} ; pmt1_aho: {results[4]}", log=True)

    await send_to_gc(f" ")
    return f"xiu20_test done"


async def xiu20_loop(meas_time_ms=1, iterations=1000):

    from predefined_tasks.pmt_adjust import set_led_current

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(False)

    await asyncio.sleep(1.0)

    await send_to_gc(f" ")
    await send_to_gc(f"; iteration ; pmt1_cnt ; pmt1_al  ; pmt1_ah  ; pmt1_alo ; pmt1_aho ;", log=True)

    op_id = 'flash_lum'
    meas_unit.ClearOperations()
    await load_xiu20_operation(op_id, meas_time_ms)
    
    for i in range(iterations):
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        await send_to_gc(f"; {i+1:9d} ; {results[0]:8d} ; {results[1]:8d} ; {results[2]:8d} ; {results[3]:8d} ; {results[4]:8d} ;", log=True)

    await send_to_gc(f" ")

    return f"xiu20_loop done"


async def xiu20_meas_time_scan(start_ms=1, stop_ms=1000, step_ms=1):

    from predefined_tasks.pmt_adjust import set_led_current

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(False)

    await asyncio.sleep(1.0)

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        await send_to_gc(f"xiu20_meas_time_scan(start_ms={start_ms}, stop_ms={stop_ms}, step_ms={step_ms}) started at {timestamp}", report=report)
        await send_to_gc(f" ")
        await send_to_gc(f"meas_time_ms ; cnt      ; alr   ; ahr   ; alo   ; aho   ; al    ; ah    ;", log=True, report=report)

        meas_time_ms_range = [t / 1e3 for t in range(round(start_ms * 1e3), round((stop_ms + step_ms) * 1e3), round(step_ms * 1e3))]

        for meas_time_ms in meas_time_ms_range:
            op_id = 'flash_lum'
            meas_unit.ClearOperations()
            await load_xiu20_operation(op_id, meas_time_ms)
            
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            al = results[1] - results[3]
            ah = results[2] - results[4]

            await send_to_gc(f"{meas_time_ms:12.3f} ; {results[0]:8d} ; {results[1]:5d} ; {results[2]:5d} ; {results[3]:5d} ; {results[4]:5d} ; {al:5d} ; {ah:5d} ;", log=True, report=report)

        await send_to_gc(f" ", report=report)

    return f"xiu20_meas_time_scan done"


async def xiu20_signal_scan(meas_time_ms=1, led_source='smu', led_channel='led1', led_current_start=0, led_current_stop=200, led_current_step=10):

    from predefined_tasks.pmt_adjust import set_led_current

    GlobalVar.set_stop_gc(False)

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(True)

    await asyncio.sleep(1.0)

    op_id = 'flash_lum'
    meas_unit.ClearOperations()
    await load_xiu20_operation(op_id, meas_time_ms)

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        await send_to_gc(f"xiu20_signal_scan(meas_time_ms={meas_time_ms}, led_source={led_source}, led_channel={led_channel}, led_current_start={led_current_start}, led_current_stop={led_current_stop}, led_current_step={led_current_step}) started at {timestamp}", report=report)
        await send_to_gc(f" ")
        await send_to_gc(f"current ; power   ; pmt1_cnt   ; pmt1_al    ; pmt1_ah    ; pmt1_alo   ; pmt1_aho   ;", log=True, report=report)

        for led_current in range(led_current_start, (led_current_stop + led_current_step), led_current_step):
                
            if GlobalVar.get_stop_gc():
                return f"flash_lum_signal_scan stopped by user"

            await set_led_current(led_current, source=led_source, channel=led_channel, led_type='green')
            await asyncio.sleep(0.5)
        
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            await send_to_gc(f"{led_current:7d} ; {flash_lum_get_led_power(led_current):7.2f} ; {results[0]:10d} ; {results[1]:10d} ; {results[2]:10d} ; {results[3]:10d} ; {results[4]:10d} ;", log=True, report=report)
        
        await meas_unit.Set_PMT_HV(False)
        await set_led_current(0, source=led_source, channel=led_channel, led_type='green')

        await send_to_gc(f" ", report=report)

    return f"xiu20_signal_scan done"


async def load_xiu20_decay_operation(op_id, window_us, window_count=1, sync_delay_ms=0, trigger=1):
    sequence = xiu20_decay_measurement(window_us, window_count, sync_delay_ms, trigger)
    meas_unit.resultAddresses[op_id] = range(0, (window_count * 5))
    await meas_unit.LoadTriggerSequence(op_id, sequence)


def xiu20_decay_measurement(window_us, window_count, sync_delay_ms, trigger):
    if (window_us < 12) or (window_us > 2**32):
        raise ValueError(f"window_us must be in the range [12, 2^32] us")
    if (window_count < 1) or (window_count > 1364):
        raise ValueError(f"window_count must be in the range [1, 1364]")
    
    full_reset_delay = 40000    # 400 us
    reset_low_delay = 1000      #  10 us
    reset_off_delay = 1000      #  10 us
    sample_offset_delay = 2     #  20 ns
    pre_cnt_window = 100        #   1 us

    window_us = round(window_us) - 1
    window_us_coarse, window_us_fine = divmod(window_us, 65536)

    sync_delay = round(sync_delay_ms * 1e5) - full_reset_delay - reset_low_delay - reset_off_delay - sample_offset_delay
    sync_delay_coarse, sync_delay_fine = divmod(sync_delay, 2**26)

    seq_gen = meas_seq_generator()

    # results = [0, pmt1_alo, pmt1_aho, pmt1_cnt, pmt1_al, pmt1_ah, ...]
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop((window_count + 1) * 3)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)

    seq_gen.SetSignals(OutputSignal.InputGatePMT1)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)

    # Wait for sync signal from function generator for normal mode
    if sync_delay_ms > 0:
        seq_gen.WaitForTriggerInput(TRF_hi=True)
    if sync_delay_coarse > 0:
        seq_gen.Loop(sync_delay_coarse)
        seq_gen.TimerWaitAndRestart(2**26)
        seq_gen.LoopEnd()
    if sync_delay_fine > 0:
        seq_gen.TimerWaitAndRestart(sync_delay_fine)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    # Reset Low + High Integrators (Reset=High, Range=High) -> Integrator Mode := Full Reset
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(reset_low_delay)
    # Reset Low Integrator (Reset=Low, Range=High) -> Integrator Mode := Integrate in High Range
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_high_range)

    seq_gen.TimerWaitAndRestart(reset_off_delay)
    # Reset Off (Reset=Low, Range=Low) -> Integrator Mode := Integrate in Low Range
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_in_low_range)

    seq_gen.TimerWaitAndRestart(sample_offset_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    # Trigger function generator for burst mode
    if trigger:
        seq_gen.SetTriggerOutput(TriggerSignal.TRF)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)

    seq_gen.Loop(window_count)

    if window_us_coarse > 0:
        seq_gen.Loop(window_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if window_us_fine > 0:
        seq_gen.Loop(window_us_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    # Analog results of previous window (last micro second of the window)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=2)

    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=3)
    
    # Last counting step of current window, then measure analog
    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)

    # Counting result of current window
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)

    seq_gen.LoopEnd()

    # Analog results of last window
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=True, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=2)

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    return seq_gen.currSequence


async def xiu20_decay_curve(window_us=12, window_count=420, sync_delay_ms=0, trigger=1):

    GlobalVar.set_stop_gc(False)

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(True)

    await asyncio.sleep(1.0)

    op_id = 'flash_lum'
    meas_unit.ClearOperations()
    await load_xiu20_decay_operation(op_id, window_us, window_count, sync_delay_ms, trigger)

    with open(report_file, 'a') as report:
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        await send_to_gc(f"xiu20_decay_curve(window_us={window_us}, window_count={window_count}, sync_delay_ms={sync_delay_ms:.5f}, trigger={trigger}) started at {timestamp}", report=report)
        await send_to_gc(f" ")

        await send_to_gc(f" ")
        await send_to_gc(f"time ; cnt        ; alr   ; ahr   ; al    ; ah    ;", log=True, report=report)

        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        for i in range(window_count + 1):
            time = round(i * window_us)
            cnt = results[i * 3 + 0]
            alr = results[i * 3 + 1]
            ahr = results[i * 3 + 2]
            al  = results[i * 3 + 1] - results[(i - 1) * 3 + 1] if i > 0 else 0
            ah  = results[i * 3 + 2] - results[(i - 1) * 3 + 2] if i > 0 else 0

            await send_to_gc(f"{time:4d} ; {cnt:10d} ; {alr:5d} ; {ahr:5d} ; {al:5d} ; {ah:5d} ;", log=True, report=report)
        
        await meas_unit.Set_PMT_HV(False)

        await send_to_gc(f" ", report=report)

    return f"xiu20_decay_curve done"


# XIU09 - Old Analog Measurement Electronics Tests #####################################################################

async def load_xiu09_operation(op_id, meas_time_ms, repeats=1, interval_ms=0.0):
    sequence = xiu09_measurement(meas_time_ms, repeats, interval_ms)
    meas_unit.resultAddresses[op_id] = range(0, (repeats * 5))
    await meas_unit.LoadTriggerSequence(op_id, sequence)


def xiu09_measurement(meas_time_ms, repeats, interval_ms):
    if (meas_time_ms > 10000):
        raise ValueError(f"meas_time_ms must be smaller or equal to 10000 ms")
    if (repeats < 1) or (repeats > 682):
        raise ValueError(f"repeats must be in the range [1, 682]")
    if (repeats > 1) and (interval_ms < meas_time_ms):
        raise ValueError(f"interval_ms must be greater or equal to meas_time_ms")
    if (interval_ms > 43980465):
        raise ValueError(f"interval_ms must be smaller or equal to 43980465 ms")

    analog_offset = 1000

    full_reset_delay = 40000    # 400 us
    conversion_delay = 1200     #  12 us
    range_switch_delay = 25     # 250 ns
    reset_switch_delay = 2000   #  20 us
    input_gate_delay = 100      #   1 us
    pre_cnt_window = 100        #   1 us
    fixed_range = 2000          #  20 us

    meas_time_us = round(meas_time_ms * 1000)
    meas_time_us_coarse, meas_time_us_fine = divmod(meas_time_us, 65536)

    dead_time = full_reset_delay + 2 * conversion_delay + range_switch_delay + reset_switch_delay + input_gate_delay + pre_cnt_window + fixed_range  # 466.25 us

    interval_delay = round(interval_ms * 1e5) - round(meas_time_us * 100) - dead_time
    interval_delay_coarse, interval_delay_fine = divmod(interval_delay, 2**26)

    seq_gen = meas_seq_generator()

    # results = [pmt1_cnt, pmt1_al, pmt1_ah, pmt1_alo, pmt1_aho, ...]
    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)
    seq_gen.Loop(repeats * 5)
    seq_gen.ClearResultBuffer(relative=True, dword=False, addrReg=0, addr=0)
    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=1)
    seq_gen.LoopEnd()

    seq_gen.SetAddrReg(relative=False, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=0)

    seq_gen.LoadDataReg(stackNotRegDst=False, dstReg=0, value=(-analog_offset & 0xFFFF))

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
    seq_gen.SetSignals(OutputSignal.HVGatePMT1)

    seq_gen.Loop(repeats)
    
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset, pmt2=AnalogControlMode.full_offset_reset)

    seq_gen.TimerWaitAndRestart(full_reset_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset, pmt2=IntegratorMode.full_reset)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1 | TriggerSignal.SamplePMT2)

    seq_gen.TimerWaitAndRestart(range_switch_delay)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.low_range_reset)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=4)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset)

    seq_gen.TimerWaitAndRestart(reset_switch_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_autorange)

    seq_gen.TimerWaitAndRestart(input_gate_delay)
    seq_gen.SetSignals(OutputSignal.InputGatePMT1)

    seq_gen.TimerWaitAndRestart(pre_cnt_window)
    seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
    if meas_time_us_coarse > 0:
        seq_gen.Loop(meas_time_us_coarse)
        seq_gen.Loop(65536)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()
        seq_gen.LoopEnd()
    if meas_time_us_fine > 0:
        seq_gen.Loop(meas_time_us_fine)
        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.PMT1, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
        seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.InputGatePMT1)
    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)

    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.read_offset)
    seq_gen.SetAnalogControl(pmt1=ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET)

    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.integrate_with_fixed_range)

    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SamplePMT1)
    seq_gen.GetPulseCounterResult(MeasurementChannel.PMT1, relative=True, resetCounter=True, cumulative=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=1)
    seq_gen.GetAnalogResult(MeasurementChannel.PMT1, isRelativeAddr=True, ignoreRange=False, isHiRange=True, addResult=True, dword=False, addrPos=0, resultPos=2)

    seq_gen.SetAddrReg(relative=True, dataNotAddrSrc=False, sign=False, stackNotRegSrc=False, srcReg=0, dstReg=0, addr=5)

    if interval_delay_coarse > 0:
        seq_gen.Loop(interval_delay_coarse)
        seq_gen.TimerWaitAndRestart(2**26)
        seq_gen.LoopEnd()
    if interval_delay_fine > 0:
        seq_gen.TimerWaitAndRestart(interval_delay_fine)
    seq_gen.LoopEnd()

    seq_gen.ResetSignals(OutputSignal.HVGatePMT1)
    seq_gen.SetIntegratorMode(pmt1=IntegratorMode.full_reset)
    seq_gen.SetAnalogControl(pmt1=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)

    return seq_gen.currSequence


async def xiu09_loop(meas_time_ms=1, iterations=1000):

    from predefined_tasks.pmt_adjust import set_led_current

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(False)

    await asyncio.sleep(1.0)

    await send_to_gc(f" ")
    await send_to_gc(f"; iteration ; pmt1_cnt ; pmt1_al  ; pmt1_ah  ; pmt1_alo ; pmt1_aho ;", log=True)

    op_id = 'flash_lum'
    meas_unit.ClearOperations()
    await load_xiu09_operation(op_id, meas_time_ms)
    
    for i in range(iterations):
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        await send_to_gc(f"; {i+1:9d} ; {results[0]:8d} ; {results[1]:8d} ; {results[2]:8d} ; {results[3]:8d} ; {results[4]:8d} ;", log=True)

    await send_to_gc(f" ")

    return f"xiu09_loop done"


async def xiu09_meas_time_scan(start_ms=1, stop_ms=1000, step_ms=1):

    from predefined_tasks.pmt_adjust import set_led_current

    await send_to_gc(f"Starting Firmware")
    await asyncio.gather(
        fmb_unit.StartFirmware(),
        eef_unit.StartFirmware(),
    )

    await meas_unit.InitializeDevice()
    await meas_unit.Set_PMT_HV(False)

    await asyncio.sleep(1.0)

    await send_to_gc(f" ")
    await send_to_gc(f"; meas_time_ms ; pmt1_cnt ; pmt1_al  ; pmt1_ah  ; pmt1_alo ; pmt1_aho ;", log=True)

    meas_time_ms_range = [t / 1e3 for t in range(round(start_ms * 1e3), round((stop_ms + step_ms) * 1e3), round(step_ms * 1e3))]

    for meas_time_ms in meas_time_ms_range:
        op_id = 'flash_lum'
        meas_unit.ClearOperations()
        await load_xiu09_operation(op_id, meas_time_ms)
        
        await meas_unit.ExecuteMeasurement(op_id)
        results = await meas_unit.ReadMeasurementValues(op_id)

        await send_to_gc(f"; {meas_time_ms:12.3f} ; {results[0]:8d} ; {results[1]:8d} ; {results[2]:8d} ; {results[3]:8d} ; {results[4]:8d} ;", log=True)

    await send_to_gc(f" ")
    return f"xiu09_meas_time_scan done"

